/* 
   Temp4dWr.cpp - example of reading from a netcdf file using the modern C++ interface.
   Based on pres_temp4d_rd.cpp from the unidata.ucar netCDF package. 
   Shortened, reading only temperature and not pressure, and made more C++-like.

   Original copyright notice:

   This is part of the netCDF package.
   Copyright 2006 University Corporation for Atmospheric Research/Unidata.
   See COPYRIGHT file for conditions of use.

   This is an example which reads some 4D pressure and temperature
   values. The data file read by this program is produced by the
   companion program pres_temp4d_wr.cpp. It is intended to illustrate
   the use of the netCDF C++ API.

   This program is part of the netCDF tutorial:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-tutorial

   Full documentation of the netCDF C++ API can be found at:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-cxx

   $Id: pres_temp4d_rd.cpp,v 1.5 2010/02/11 22:36:42 russ Exp $
*/

#include <iostream>
#include <netcdf>
#include <rarray>

int main()
{
    // This is the name of the data file we will create. 
    const std::string FILE_NAME = "Temp4d.nc";

    // The dimensions will be read from the netcdf fle, but are
    // defined here to check if the file was written correctly.
    const int NLVL = 2;
    const int NLAT = 6;
    const int NLON = 12;
    const int NREC = 2;
    
    // Names of things. 
    const std::string LAT_NAME = "latitude";
    const std::string LON_NAME = "longitude";
    const std::string TEMP_NAME = "temperature";
    
    // These are used to construct some example data:
    const float SAMPLE_TEMP = 9.0;
    const float START_LAT = 25.0;
    const float START_LON = -125.0;

    // open the file:
    auto temperatureFile = std::make_unique<netCDF::NcFile>(FILE_NAME,
                                                            netCDF::NcFile::read);
    
    // Get the latitude and longitude variables and read data:
    netCDF::NcVar latVar = temperatureFile->getVar(LAT_NAME);
    netCDF::NcVar lonVar = temperatureFile->getVar(LON_NAME);
    if (latVar.isNull()) return 1;
    if (lonVar.isNull()) return 1;
    if (latVar.getDimCount() != 1) return 1;
    if (lonVar.getDimCount() != 1) return 1;
    size_t nlat = latVar.getDim(0).getSize();
    size_t nlon = lonVar.getDim(0).getSize();
    if (nlat != NLAT) return 1;
    if (nlon != NLON) return 1;
    
    // Check the coordinate variable data:
    rarray<float,1> lats(nlat);   
    latVar.getVar(lats.data());
    for (int lat: xrange(nlat)) 
        if (lats[lat] != START_LAT + 5. * lat) 
            return 1;
    rarray<float,1> lons(nlon);
    lonVar.getVar(lons.data());
    for (int lon: xrange(nlon)) 
        if (lons[lon] != START_LON + 5. * lon) 
            return 1;
   
    // Get the temperature variables and read data one time step at a time:
    netCDF::NcVar tempVar = temperatureFile->getVar(TEMP_NAME);
    if (tempVar.isNull()) return 1; 
    if (tempVar.getDimCount() != 4) return 1;
    size_t nrec = tempVar.getDim(0).getSize();
    if (nrec != NREC) return 1;
    if (tempVar.getDim(1).getSize() != NLVL) return 1;
    if (tempVar.getDim(2).getSize() != lats.size()) return 1;
    if (tempVar.getDim(3).getSize() != lons.size()) return 1;

    rarray<float,3> temp_in(tempVar.getDim(1).getSize(), lats.size(), lons.size());

    for (int rec: xrange(nrec)) {
        // Read the data one record at a time:
        tempVar.getVar({rec,0,0,0},
                       {1,temp_in.extent(0),temp_in.extent(1),temp_in.extent(2)},
                       temp_in.data());
        // Check:
        int i=0;
        for (float& temp: temp_in) 
            if (temp != (SAMPLE_TEMP + i++))
                return 1;
    } 

    // Note: When the unique pointer temperatureFile does out of scope
    // or is reset to a nullptr, the file is automatically closed.
    temperatureFile.reset(nullptr);
    
    std::cout << "*** SUCCESS reading example file " << FILE_NAME << "!\n";
    return 0;
}
