/* 
 *  lj.c
 *
 *  Example of a Lennard-Jones simulation in C
 *
 *  Input (from standard input, separate line in a file, or command-line arguments): 
 *
 *   a folder name for output (1024 chars max, no spaces or tabs)
 *   substance (one of the following: LJ Ar CO2 He Kr Ne N2 O2 Xe)
 *   initial temperature (standard deviation of the velocities)
 *   density of the system
 *   number of particles 
 *   runtime
 *   time step 
 *   random number generator seed
 *   equilibration time 
 *   traject
 *
 *  Output:
 *
 *   1. A file "output.dat" is written in the output folder, containing lines with:
 *
 *       time, total energy, potential energy, kinetic energy, temperature, pressure
 *
 *      A final line will have the averages of total energy, potential
 *      energy, kinetic energy, temperature, pressure.
 *
 *      If the output folder name starts with '-', output is written
 *      to standard output instead.
 *
 *   2. A trajectory file output.xsf is written too. Not created if
 *      the output folder name starts with '-', or if the traject
 *      parameter is 'F' or 'False'.
 *
 *   3. A JSON file report.json is written in the output folder at the
 *      end of the simulation, containing the parameters and the
 *      averaged measured quantities (e.i., energies, temperature,
 *      pressure) over the simulation run.
 *
 *  Notes:
 *
 *  - To compile: gcc -O3 -std=c99 -D_XOPEN_SOURCE=500 lj.c -o lj -lm
 *
 *  - To run: 
 *      ./lj < argon.in
 *    or: 
 *      ./lj argon.in
 *    or: 
 *      ./lj argonoutput Ar 100 2.5 500 12 0.005 17 3.0 True
 *
 *    where input parameters are listed in the file "lj.in" o on the
 *    command line.
 *
 *  - All reported energies values are divided by the number of
 *    particles N.
 *
 *  - This program is intended for small systems, so no attempt was
 *    made to implement cell divisions or Verlet neighbor lists.
 *
 *  Ramses van Zon, 13 November 2008 
 *   Feb 27 2012: simplified
 *   Mar 28 2012: simpler rand function instead of rand48, and
 *                de-c99'ed for mingw portability.
 *   Oct 28 2013: added pressure calcuation
 *   Jun 19 2017: clean up and added other units
 *   Dec 4  2021: separate file for final averages
 */

#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <math.h>
#include <sys/stat.h>
#include "substances.h"

typedef struct 
{
   int     N;           /* number of particles */
   double  rho;         /* density of the system */
   double  L;           /* length of the box */
   double  dt;          /* duration of single time step */
   double  runtime;     /* how long to run */
   long    seed;        /* random seed */
   double  equilTime;   /* when to start measuring energies etc. */
   double  Tinit;       /* initial kinetic temperature */
   double  V;           /* volume */
   char    outdir[PATH_MAX];/* name of output directory */
   char    substance[MAX_SUBSTANCE_NAME_LEN]; /* what substance */
   char    traject;     /* should a traject be written out? T or F */
   lj_parameters_t substdef; /* substance definition */
} Parameters;

typedef struct 
{
   double  K;           /* kinetic energy */
   double  U;           /* potential energy */
   double  virial;      /* virial, for pressure calculation */
   double  P;           /* pressure */
   double  H;           /* total energy */
   double  T;           /* kinetic temperature */
   FILE*   file;        /* output file */
   char    filepath[PATH_MAX]; /* full path of the output file */ 
   FILE*   xsffile;     /* xsf output file */
} Properties;

/* structure for the properties of one atoms */
typedef struct 
{
   double  rx, ry, rz;  /* position */
   double  px, py, pz;  /* momentum */
   double  fx, fy, fz;  /* force */

} Atom;

/* function to set up cubic lattice */
double latticex, latticey, latticez;
void makeLatticePosition(const Parameters* parm, double a)
{
   static int i = 0;
   static int j = 0;
   static int k = 0;
   latticex = i*a - 0.5*parm->L;
   latticey = j*a - 0.5*parm->L;
   latticez = k*a - 0.5*parm->L;
   i++;
   if ( i*a > parm->L - 1.0e-6 )        /* next point outside box? */
   {
      i = 0;                            /* then put back */
      j++;                              /* and move to next y level */

      if ( j*a > parm->L - 1.0e-6 )     /* outside box? */
      {
         j = 0;                         /* then put back */
         k++;                           /* and move to next z level */

         if ( k*a > parm->L - 1.0e-6 )  /* outside box? */
         {
            i = 0;                      /* then back to the start */
            j = 0;
            k = 0;
         }
      }
   }
}

/* function dealing with periodic boundary conditions: */
/* increases or decreases *u until -L/2 <= *u < L/2 */
double makePeriodic(const Parameters* parm, double u)
{
   if ( u < -0.5*parm->L ) {
      u += parm->L;
   }
   if ( u >= 0.5*parm->L ) {
      u -= parm->L;
   }
   return u;
} 

/* function to compute forces */
void computeForces(const Parameters* parm, Properties* prop, Atom atoms[])
{
   /* initialize energy and forces to zero */   
   prop->U = 0;
   prop->virial = 0;

   for ( int i = 0; i < parm->N; i++ ) {
      atoms[i].fx = 0;
      atoms[i].fy = 0;
      atoms[i].fz = 0;
   }

   /* determine interaction for each pair of particles (i,j) */
   for ( int i = 0; i < parm->N-1; i++ )
   {
      for ( int j = i+1; j < parm->N; j++ ) 
      {
         double  rc = 3.0;      /* cutoff radius */
         double  r2, r2i, r6i;  /* distance, r^2, r^{-2} and r^{-6} */
         double  fij;           /* force multiplier */
         double  eij;           /* potential energy between i and j */

         /* determine distance in periodic system */
         double dx = makePeriodic(parm, atoms[i].rx - atoms[j].rx);
         double dy = makePeriodic(parm, atoms[i].ry - atoms[j].ry);
         double dz = makePeriodic(parm, atoms[i].rz - atoms[j].rz);
         r2 = dx*dx + dy*dy + dz*dz; 

         /* note : using square distance saves taking a sqrt */
         if ( r2 < rc*rc ) 
         {
            double  irc6 = 1.0/(rc*rc*rc*rc*rc*rc);
            double  ec = 4.0*irc6*(irc6-1);/* potential energy at radius rc*/

            r2i = 1/r2;
            r6i = r2i*r2i*r2i;
            fij = 48*r2i*r6i*(r6i-0.5);
            eij = 4*r6i*(r6i-1) - ec;

            atoms[i].fx += fij*dx;
            atoms[i].fy += fij*dy;
            atoms[i].fz += fij*dz;
            atoms[j].fx -= fij*dx;
            atoms[j].fy -= fij*dy;
            atoms[j].fz -= fij*dz;
            prop->U += eij;
            prop->virial += fij*r2;
         }
      }
   }
}

double drand() 
{
   /* don't use this random number generator! */
   return (double)rand()/INT_MAX;
}

/* function for gaussian random variables */
double gaussian()
{
   const  double pi = 3.14159265358979323846264338327950288;
   static int    have = 0;
   static double x2;
   double fac, y1, y2, x1;
 
   if ( have == 1 )  /* already one available ? */
   {
      have = 0;
      return x2;
   } 
   else 
   {
      /* generate a pair of random variables */
      y1  = drand();
      y2  = drand();
      fac = sqrt(-2*log(y1));
      have = 1;
      x1 = fac*sin(2*pi*y2); /* x1 and x2 are now gaussian */
      x2 = fac*cos(2*pi*y2); /* so store one */
      return x1;               /* and return the other. */
   }
}

/* function to initialize the system */
void initialize(const Parameters* parm, Properties* prop, Atom atoms[])
{
   double  scale, a;

   /* generate positions */
   a = parm->L/(int)(cbrt(parm->N)+0.99999999999); /* lattice distance */

   for ( int i = 0; i < parm->N; i++ ) 
   {
      makeLatticePosition(parm, a);
      atoms[i].rx = latticex;
      atoms[i].ry = latticey;
      atoms[i].rz = latticez;
   }

   /* generate momenta */
   srand(parm->seed);  /* initialized the random number generator used in gaussian */
   scale   = sqrt(parm->Tinit);
   prop->K = 0;

   for ( int i = 0; i < parm->N; i++ ) 
   {
      atoms[i].px = scale*gaussian();
      atoms[i].py = scale*gaussian();
      atoms[i].pz = scale*gaussian();
      prop->K += atoms[i].px*atoms[i].px 
               + atoms[i].py*atoms[i].py 
               + atoms[i].pz*atoms[i].pz;
   }

   /* compute instantaneous kinetic temperature and energy */
   prop->T = prop->K/(3*parm->N);
   prop->K /= 2;

   /* compute force and potential energy U */
   computeForces(parm, prop, atoms);

   /* compute total energy */
   prop->H = prop->U + prop->K;

   /* compute pressure */
   prop->P = (2*prop->K+prop->virial)/(3*parm->V);

   /* save in the parm->outdir/output.dat file, unless parm->outdir starts with '-'. */
   if (parm->outdir[0] == '-')
   {
       prop->file = stdout;
       strcpy(prop->filepath, "<STDOUT>");
   }
   else
   {
       /* create folder parm->outdir if it does not exist */
       struct stat sb;
       if (! ( stat(parm->outdir, &sb) == 0 && S_ISDIR(sb.st_mode) ) )
           mkdir(parm->outdir, 0777);
       /* then open file 'output.dat' inside that folder */
       const char filename[] = "output.dat";
       char fullfilename[PATH_MAX + 1 + sizeof(filename)];
       strncpy(fullfilename, parm->outdir, PATH_MAX);
       fullfilename[strlen(parm->outdir)] = '/';
       strcpy(fullfilename+strlen(parm->outdir) + 1, filename);
       prop->file = fopen(fullfilename, "w");
       if ( ! prop->file ) {
           fprintf(stderr, "Cannot open output file %s:\n", fullfilename);
           exit(1);
       }
       if ( realpath(fullfilename, prop->filepath) != prop->filepath ) {
           fprintf(stderr, "Cannot find path to output file %s:\n", fullfilename);
           exit(1);
       }
   }

   /* report results */ 
   fprintf(prop->file, "# time   E(kJ/mol) U(kJ/mol) K(kJ/mol) T(K)     P(MPa)\n");
   
   if (parm->equilTime <= 0.0) 
   {
      fprintf(prop->file,
              "%8.6f %8.6f %8.6f %8.6f %8.6f %8.6f\n", 
              0., 
              energy_to_physical(prop->H/parm->N, &parm->substdef),
              energy_to_physical(prop->U/parm->N, &parm->substdef), 
              energy_to_physical(prop->K/parm->N, &parm->substdef), 
              temperature_to_physical(prop->T, &parm->substdef), 
              pressure_to_physical(prop->P, &parm->substdef));
   }

   /* for trajectory saving, open xsf file */
   if (parm->outdir[0] != '-' && parm->traject == 'T') {
       const char xsffilename[] = "output.xsf";
       char fullxsffilename[PATH_MAX + 1 + sizeof(xsffilename)];
       strncpy(fullxsffilename, parm->outdir, PATH_MAX);
       fullxsffilename[strlen(parm->outdir)] = '/';
       strcpy(fullxsffilename+strlen(parm->outdir) + 1, xsffilename);
       prop->xsffile = fopen(fullxsffilename, "w");
       if ( ! prop->xsffile ) exit(1);
       double primL = length_to_physical(parm->L, &parm->substdef);
       fprintf(prop->xsffile,
               "PRIMVEC\n"
               "%f 0.0 0.0\n"
               "0.0 %f 0.0\n"
               "0.0 0.0 %f\n",
               primL, primL, primL);
       fprintf(prop->xsffile,
               "ANIMSTEPS 10000000\n"); /* comfortable large */
   } else {
       prop->xsffile = NULL;
   }
}

/* Verlet integration step */
void integrateStep(const Parameters* parm, Properties* prop, Atom atoms[])
{
   /* half-force step */
   for ( int i = 0; i < parm->N; i++ ) 
   {
      atoms[i].px += 0.5*parm->dt*atoms[i].fx;
      atoms[i].py += 0.5*parm->dt*atoms[i].fy;
      atoms[i].pz += 0.5*parm->dt*atoms[i].fz;
   }

   /* full free motion step */
   for ( int i = 0; i < parm->N; i++ ) 
   {
      atoms[i].rx += parm->dt*atoms[i].px;
      atoms[i].ry += parm->dt*atoms[i].py;
      atoms[i].rz += parm->dt*atoms[i].pz;
   }

   /* positions were changed, so recompute the forces */
   computeForces(parm, prop, atoms);

   /* final force half-step */
   prop->K = 0;

   for ( int i = 0; i < parm->N; i++ ) 
   {
      atoms[i].px += 0.5*parm->dt*atoms[i].fx;
      atoms[i].py += 0.5*parm->dt*atoms[i].fy;
      atoms[i].pz += 0.5*parm->dt*atoms[i].fz;
      prop->K += atoms[i].px*atoms[i].px 
               + atoms[i].py*atoms[i].py 
               + atoms[i].pz*atoms[i].pz;
   }

   /* finish computing T, K, H */
   prop->T = prop->K/(3*parm->N);
   prop->K /= 2;
   prop->H = prop->U + prop->K;
}

/* rescaling of velocities for target temperature */
void rescaleVelocities(const Parameters* parm, Properties* prop, Atom atoms[])
{
   double scale = sqrt(parm->Tinit / prop->T);
   for ( int i = 0; i < parm->N; i++ ) 
   {
      atoms[i].px *= scale;
      atoms[i].py *= scale;
      atoms[i].pz *= scale;
   }   
}


/* write to xsf file */
void writeXSF(const Parameters* parm, const Properties* prop, const Atom atoms[])
{   
   static int n = 0;
   if (prop->xsffile) {
      n++;
      fprintf(prop->xsffile, "ATOMS %d\n", n);
      for ( int i = 0; i < parm->N; i++ )
      {
         double x = length_to_physical( makePeriodic(parm, atoms[i].rx), &parm->substdef );
         double y = length_to_physical( makePeriodic(parm, atoms[i].ry), &parm->substdef );
         double z = length_to_physical( makePeriodic(parm, atoms[i].rz), &parm->substdef );
         fprintf(prop->xsffile,
                 "%d %f %f %f\n",
                 parm->substdef.atomic_number, x, y, z);
      }
   }
}

/* integration and measurement */
void run(const Parameters* parm)
{
   Atom*       atoms      = malloc(parm->N*sizeof(Atom)); 
   const int   equilSteps = (int)(parm->equilTime/parm->dt + 0.5);
   const int   numSteps   = (int)(parm->runtime/parm->dt + 0.5);
   Properties* prop       = malloc(sizeof(Properties));

   /* draw initial conditions */
   initialize(parm, prop, atoms);

   /* report parameters */
   fprintf(prop->file,
           "#Substance=%s N=%d L=%lf T=%lf runtime=%lf dt=%lf seed=%ld equil=%lf\n",
           parm->substdef.name, parm->N, 
           length_to_physical(parm->L, &parm->substdef), 
           temperature_to_physical(parm->Tinit, &parm->substdef), 
           time_to_physical(parm->runtime, &parm->substdef),
           time_to_physical(parm->dt, &parm->substdef), 
           parm->seed, 
           time_to_physical(parm->equilTime, &parm->substdef));

   /* perform equilibration steps */
   for ( int count = 0; count < equilSteps; count++ ) 
   {
      /* perform integration step */
      integrateStep(parm, prop, atoms);

      /* adjust to desired temperature */
      rescaleVelocities(parm, prop, atoms);
   }

   /* perform regular steps (with output and averaging) */
   double avH = 0.0;
   double avU = 0.0;
   double avK = 0.0;
   double avT = 0.0;
   double avP = 0.0;
   for ( int count = equilSteps ; count < numSteps; count++ ) 
   {
      /* perform integration step */
      integrateStep(parm, prop, atoms);

      /* compute pressure */
      prop->P = (2*prop->K+prop->virial)/(3*parm->V);  

      /* report */       
      fprintf(prop->file,
              "%8.6f %8.6f %8.6f %8.6f %8.6f %8.6f\n", 
              time_to_physical((count+1)*parm->dt, &parm->substdef),
              energy_to_physical(prop->H/parm->N, &parm->substdef),
              energy_to_physical(prop->U/parm->N, &parm->substdef), 
              energy_to_physical(prop->K/parm->N, &parm->substdef), 
              temperature_to_physical(prop->T, &parm->substdef), 
              pressure_to_physical(prop->P, &parm->substdef));

      /* write trajectory */
      writeXSF(parm, prop, atoms);

      /* accumulate */
      avH +=  prop->H;
      avU +=  prop->U;
      avK +=  prop->K;
      avT +=  prop->T;
      avP +=  prop->P;
   }

   if (numSteps > equilSteps)
   {
      /* compute averages */
      avH /= (numSteps-equilSteps)*parm->N;
      avU /= (numSteps-equilSteps)*parm->N;
      avK /= (numSteps-equilSteps)*parm->N;
      avT /= (numSteps-equilSteps);
      avP /= (numSteps-equilSteps);
      /* output and full report */ 
      fprintf(prop->file,
              "#<>: %s %s %8.6f %8.6f %8.6f %8.6f %8.6f\n", 
              parm->outdir, 
              parm->substance, 
              energy_to_physical(avH, &parm->substdef),
              energy_to_physical(avU, &parm->substdef),
              energy_to_physical(avK, &parm->substdef),
              temperature_to_physical(avT, &parm->substdef),
              pressure_to_physical(avP, &parm->substdef));
       const char filename[] = "report.json";
       char fullfilename[PATH_MAX + 1 + sizeof(filename)];
       strncpy(fullfilename, parm->outdir, PATH_MAX);
       fullfilename[strlen(parm->outdir)] = '/';
       strcpy(fullfilename+strlen(parm->outdir) + 1, filename);
       FILE* f = fopen(fullfilename, "w");
       if ( ! f ) {
           fprintf(stderr, "# Writing json file to stdout\n");
           f = stdout;
       }
       fprintf(f,
              "{\n"
              "    \"outdir\": \"%s\",\n"
              "    \"parameters\": {\n"
              "        \"substance\"  : \"%s\",\n"
              "        \"density\"    : { \"value\": %lf, \"unit\": \"%s\" },\n"
              "        \"boxlength\"  : { \"value\": %lf, \"unit\": \"%s\" },\n"
              "        \"number\"     : { \"value\": %d, \"unit\": \"1\"},\n"
              "        \"temperature\": { \"value\": %lf, \"unit\": \"%s\" },\n"
              "        \"runtime\"    : { \"value\": %lf, \"unit\": \"%s\" },\n"
              "        \"timestep\"   : { \"value\": %lf, \"unit\": \"%s\" },\n"
              "        \"seed\"       : { \"value\": %ld, \"unit\": \"1\"},\n"
              "        \"equiltime\" : { \"value\": %lf, \"unit\": \"%s\" }\n"
              "    },\n"
              "    \"measurements\": {\n"
              "        \"energy\":   { \"total\":     { \"value\": %lf, \"unit\": \"%s\" },\n"
              "                      \"potential\": { \"value\": %lf, \"unit\": \"%s\" },\n"
              "                      \"kinetic\":   { \"value\": %lf, \"unit\": \"%s\" }\n"
              "                    },\n"
              "        \"temperature\": { \"value\": %lf, \"unit\": \"%s\" },\n"
              "        \"pressure\": { \"value\": %lf, \"unit\": \"%s\" }\n"                      
              "    },\n"
              "    \"data_locations\": {\n"
              "        \"output\": \"%s\"\n"
              "    }\n"
              "}\n",
              parm->outdir,
              parm->substance,
              density_to_physical(parm->rho, &parm->substdef), physical_density_unit,
              length_to_physical(parm->L, &parm->substdef), physical_length_unit,
              parm->N,
              temperature_to_physical(parm->Tinit, &parm->substdef), physical_temperature_unit,
              time_to_physical(parm->runtime, &parm->substdef), physical_time_unit,
              time_to_physical(parm->dt, &parm->substdef), physical_time_unit,
              parm->seed,
              time_to_physical(parm->equilTime, &parm->substdef), physical_time_unit,
              energy_to_physical(avH, &parm->substdef), physical_energy_unit,
              energy_to_physical(avU, &parm->substdef), physical_energy_unit,
              energy_to_physical(avK, &parm->substdef), physical_energy_unit,
              temperature_to_physical(avT, &parm->substdef), physical_temperature_unit,
              pressure_to_physical(avP, &parm->substdef), physical_pressure_unit,
              prop->filepath
              );
       if (f != stdout) {
           fclose(f);
       }
   }

   if (prop->file && prop->file != stdout && prop->file != stderr)
       fclose(prop->file);
   if (prop->xsffile)
       fclose(prop->xsffile);
   free(atoms);
   free(prop);
}

/* main program */
int simulate_from_parameter_file(FILE* f)
{
   /* read parameters from file */
   Parameters parm;
   int n;
   /* tag/id */
   n = fscanf(f, "%s", parm.outdir);
   if ( n == EOF ) return 1;
   fprintf(stderr,"outdir:       %s\n", parm.outdir);
   /* substance */
   n = fscanf(f,"%s", parm.substance);
   if ( n == EOF ) return 1;
   fprintf(stderr,"Substance:    %s\n", parm.substance);
   /* initial temperature  */
   n = fscanf(f,"%lf", &parm.Tinit);
   if ( n == EOF ) return 1;
   fprintf(stderr,"Initial.temp: %.3f\n", parm.Tinit);
   /* density */
   n =  fscanf(f,"%lf", &parm.rho);  
   if ( n == EOF ) return 1;
   fprintf(stderr,"Density:      %.3f\n", parm.rho);
   /* number of particles; allow odd formats like 343.0 */
   double N_as_double;
   n = fscanf(f,"%lf\n", &N_as_double);
   if ( n == EOF ) return 1;
   parm.N = N_as_double;
   fprintf(stderr,"Number:       %d\n", parm.N);
   /* runtime */
   n = fscanf(f,"%lf", &parm.runtime);
   if ( n == EOF ) return 1;
   fprintf(stderr,"Runtime:      %.3f\n", parm.runtime);
   /* time step */
   n = fscanf(f,"%lf", &parm.dt);
   if ( n == EOF ) return 1;
   fprintf(stderr,"Time.step:    %.3f\n", parm.dt);
   /* random seed; allow odd formats like 13.0 */
   double seed_as_double;   
   n = fscanf(f,"%lf", &seed_as_double);
   if ( n == EOF ) return 1;
   parm.seed = seed_as_double;
   fprintf(stderr,"Random.seed:  %ld\n", parm.seed);
   /* equalibration time */
   n = fscanf(f,"%lf", &parm.equilTime); 
   if ( n == EOF ) parm.equilTime = 0.0;
   fprintf(stderr,"equil.time: %.3f\n", parm.equilTime);
   /* trajectory output ? */
   char traject[100];   
   n = fscanf(f,"%s", traject);
   if ( n == EOF ) {
       parm.traject = 'T';
   } else {
       if ( traject[0] == 'F' || traject[0] == 'f' || traject[0] == 'N' || traject[0] == 'n' ) {
	   parm.traject = 'F';
       } else {
	   parm.traject = 'T';
       }
   }
   fprintf(stderr,"trajectory:   %c\n", traject[0]);
   /* convert read-in parameters */
   parm.substdef  = find_conversion(parm.substance);
   parm.Tinit     = temperature_to_LJ(parm.Tinit,     &parm.substdef);
   parm.rho       = density_to_LJ    (parm.rho,       &parm.substdef);
   parm.runtime   = time_to_LJ       (parm.runtime,   &parm.substdef);
   parm.dt        = time_to_LJ       (parm.dt,        &parm.substdef);
   parm.equilTime = time_to_LJ       (parm.equilTime, &parm.substdef);

   /* set default for equilTime */
   if (parm.equilTime == 0.0) 
   {
      parm.equilTime = parm.runtime / 2.0;
   }

   /* determine size of cubic box */
   parm.L = cbrt(parm.N/parm.rho);
   parm.V = parm.L*parm.L*parm.L;

   /* run the simulation */
   run(&parm); 

   return 0;
}

void help(FILE* f) {
    fprintf(f,
            "\nlj - Molecular simulation of particles with Lennard-Jones interactions\n\n"
            "Usages:\n"
            "  lj -h\n"
            "  lj INFILE\n"
            "  lj outdir subst T rho N runtime dt seed equil traject\n"
            "  lj < INIFILE\n\n"
            "When using an INIFILE, it should contain values for 'outdir', 'subst', 'T', 'rho', 'N', 'runtime', 'dt', 'seed', 'equil', and 'traject' on separate lines.\n\n"
            "These parameters have the following meaning:\n"

            "  outdir     folder for output\n"
            "  subst      substance (LJ Ar CO2 He Kr Ne N2 O2 or Xe)\n"
            "  T          initial temperature\n"
            "  rho        density\n"
            "  N          number of particles\n"
            "  runtime    simulation time\n"
            "  dt         time step\n"
            "  seed       random seed to initialize velocities\n"
            "  equil      equilibration time\n"
            "  traject    T or F, controls if trajectory is written out\n\n"
            );
}

int main(int argc, char** argv) 
{
   /* Interpret command-line argument  
      - 0 arguments means read parameters from stdin
      - 1 argument is a filename with the parameters
      - multiple arguments are the parameters themselves.
   */
   FILE* f = NULL;

   if ( argc == 1 )  
   {
      fprintf(stderr, "Warning: reading from standard input\n");
      f = stdin;
   }
   else if ( argc == 2 ) 
   {
      if ( strcmp(argv[1],"-h") == 0 )
      {
          help(stdout);
          return 0;
      }
      if ( !(f = fopen(argv[1],"r")) ) 
      {
         fprintf(stderr, "Could not open file %s\n", argv[1]);
         return 1;
      }
   } 
   else
   {
      if (argc != 11) {
          fprintf(stderr,"ERROR: Incorrect parameters\n");
          help(stderr);
          return 2;
      }           
      if ( f = tmpfile() )
      {
         for ( int i = 1; i < argc; i++ )
         {
            fprintf(f, "%s\n", argv[i]);
         }
         rewind(f);      
      } 
      else 
      {
         fprintf(stderr, "Could not create temporary file\n");
      } 
   } 

   /* If file with parameters ready, perform simulation */
   int exitcode = 0;
   if ( f != NULL ) 
   {
      exitcode = simulate_from_parameter_file(f);
      if ( f != stdin ) fclose(f);
   } 
   else 
   {
      exitcode = 1;
   }
   
   return exitcode;

}

