/* 
 * substances.h - Header file with Lennard-Jones parameter definitions
 *                of various substances.
 */
#ifndef SUBSTANCES_H
#define SUBSTANCES_H

#include <string.h>
#define MAX_SUBSTANCE_NAME_LEN 32

/* structure to hold conversion parameters */
typedef struct 
{
   char   name[MAX_SUBSTANCE_NAME_LEN];
   double sigma_in_angstrom;
   double epsilon_in_kbK;
   double mass_in_u;
   int    atomic_number;
} lj_parameters_t;

const char physical_density_unit[]     = "mol/l";
const char physical_length_unit[]      = "Å";
const char physical_temperature_unit[] = "K";
const char physical_time_unit[]        = "ps";
const char physical_energy_unit[]      = "kJ/mol";
const char physical_pressure_unit[]    = "MPa";
const double kbK   = 120.274; /* kb K in units of kJ/mol */
const double ps    = 1.1;     /* picosecond conversion */
const double MPa   = 13.81;   /* megapascal conversion */
const double mol_l = 1660;    /* mol per liter conversion */
/* sigma and epsilon parameters were taken from J.O. Hirshfelder and
   C.F. Curtiss, "Molecular Theory of Gases and Liquids," Wiley, New
   York, 1954, p1110ff. */

const lj_parameters_t lj_parameters[] = 
{
   /* Substance sigma(A)  epsilon(J/kb/K) mass(g/mol)  atomic-number*/
   { "LJ",       1.00,         1.00,         1.00,      1},
   { "Ar",       3.4,        120.,          39.95,     18},
   { "CO2",      4.5,        189.,          44.01,     21}, /* 21=Scandium's weight is close */
   { "He",       2.56,        10.22,         4.00,      2},
   { "Kr",       3.6,         58.8,         83.80,     36},
   { "Ne",       2.75,        35.,          20.18,     10},
   { "N2",       3.7,         95.,          28.01,     14}, /* 14=Silicon's weight is close */ 
   { "O2",       3.5,        118.,          32.00,     16}, /* S=Sulfer's weight is close */
   { "Xe",       4.1,         87.,         131.29,     54}
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static lj_parameters_t find_conversion(const char* search_name)
{
   int n = sizeof(lj_parameters)/sizeof(lj_parameters_t);

   for ( int i = 0 ; i < n; i++ )
   {
      if (strncmp(lj_parameters[i].name, search_name, MAX_SUBSTANCE_NAME_LEN)==0)
         return lj_parameters[i];
   }

   return lj_parameters[0];
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double length_to_physical(double lLJ, const lj_parameters_t* ljp) 
{
   /* convert Lennard-Jones length units to physical units (A) */
   return lLJ * ljp->sigma_in_angstrom;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double length_to_LJ(double lPHYS, const lj_parameters_t* ljp) 
{
   /* convert physical length units (A) to LJ units */
   return lPHYS / ljp->sigma_in_angstrom;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double energy_to_physical(double eLJ, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of energy to physical units (kJ/mol) */
   return eLJ * ljp->epsilon_in_kbK / kbK;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double energy_to_LJ(double ePHYS, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of energy to physical units (kJ/mol) */
   return ePHYS / ljp->epsilon_in_kbK * kbK;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double temperature_to_physical(double TLJ, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (kJ/mol) */
   return TLJ * ljp->epsilon_in_kbK;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double temperature_to_LJ(double TPHYS, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (kJ/mol) */
   return TPHYS / ljp->epsilon_in_kbK;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double time_to_physical(double tLJ, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (ps) */
   return tLJ * sqrt(ljp->mass_in_u/ljp->epsilon_in_kbK) * ljp->sigma_in_angstrom * ps;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double time_to_LJ(double tPHYS, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (ps) */
   return tPHYS * sqrt(ljp->epsilon_in_kbK/ljp->mass_in_u) / ljp->sigma_in_angstrom / ps;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double pressure_to_physical(double pLJ, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (ps) */
   return pLJ * ljp->epsilon_in_kbK/pow(ljp->sigma_in_angstrom,3) * MPa;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double pressure_to_LJ(double pPHYS, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (ps) */
   return pPHYS * pow(ljp->sigma_in_angstrom,3) / (ljp->epsilon_in_kbK * MPa);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double density_to_physical(double rhoLJ, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (ps) */
   return rhoLJ / pow(ljp->sigma_in_angstrom,3) * mol_l;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static double density_to_LJ(double rhoPHYS, const lj_parameters_t* ljp) 
{
   /* convert lennard-Jones units of temperature to physical units (ps) */
   return rhoPHYS * pow(ljp->sigma_in_angstrom,3) / mol_l;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#endif

/* 
=============================
APPENDIX: Table of LJ Units
=============================

Physical quantity      Unit       Value for Ar
-----------------------------------------------
length                 σ          3.40e-10 m
energy                 ε          1.65e-21 J
mass                   m          6.69e-26 kg 
time                   σ(m/ε)^½   2.17e−12 s
velocity               (ε/m)^½    1.57e2 m/s
force                  ε/σ        4.85e−12 N
pressure               ε/σ³       4.20e7 N/m²
temperature            ε/kB       120 K
-----------------------------------------------

*/
