/* 
   Temp4dWr.cpp - example of writing to netcdf using the modern c++ interface.
   Based on pres_temp_4D_wr.cpp from the unidata.ucar netCDF package. 
   Shortened, writing only temperature and not pressure, and made more c++-like.

   Original copyright notice:

   This is part of the netCDF package.
   Copyright 2006 University Corporation for Atmospheric Research/Unidata.
   See COPYRIGHT file for conditions of use.

   This is an example program which writes some 4D pressure and
   temperatures. This example demonstrates the netCDF C++ API. 

   This is part of the netCDF tutorial:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-tutorial

   Full documentation of the netCDF C++ API can be found at:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-cxx

   $Id: pres_temp_4D_wr.cpp,v 1.6 2010/02/11 22:36:42 russ Exp $
*/

#include <netcdf>
#include <iostream>
#include <rarray>

int main()
{
    // This is the name of the data file we will create. 
    const std::string FILE_NAME = "Temp4d.nc";

    // We are writing 4D data, a 2 x 6 x 12 lvl-lat-lon grid, with 2
    // timesteps of data.
    const int NLVL = 2;
    const int NLAT = 6;
    const int NLON = 12;
    const int NREC = 2;

    // Names of things:
    const std::string LVL_NAME = "level";
    const std::string LAT_NAME = "latitude";
    const std::string LON_NAME = "longitude";
    const std::string REC_NAME = "time";
    const std::string TEMP_NAME = "temperature";
    const std::string UNITS_NAME = "units";
    
    // These are used to construct some example data:
    const float SAMPLE_TEMP = 9.0;
    const float START_LAT = 25.0;
    const float LAST_LAT  = 50.0;
    const float START_LON = -125.0;
    const float LAST_LON  = -70.0;
    
    // For the units attributes:
    const std::string DEGREES_EAST =  "degrees_east";
    const std::string DEGREES_NORTH = "degrees_north";
    const std::string TEMP_UNITS = "celsius";
    const std::string LAT_UNITS = "degrees_north";
    const std::string LON_UNITS = "degrees_east";
    
    // Program variables to hold the data we will write out. We will
    // write latitude and longitude fields. , but we will only need
    // enough space to hold one timestep of data; one record.
    rarray<float,1> lats = linspace(START_LAT, LAST_LAT, NLAT);
    rarray<float,1> lons = linspace(START_LON, LAST_LON, NLON);
    rarray<float,3> temp_out(NLVL, lats.size(), lons.size());
    int i=0;
    for (float& temp: temp_out) 
        temp = SAMPLE_TEMP + i++; 
    
    // Create the file.
    auto temperatureFile = std::make_unique<netCDF::NcFile>(FILE_NAME,
                                                            netCDF::NcFile::replace);

    // Define the dimensions. NetCDF will hand back an ncDim object for each.
    netCDF::NcDim recDim = temperatureFile->addDim(REC_NAME);  // unlimited, because no size is given!
    netCDF::NcDim lvlDim = temperatureFile->addDim(LVL_NAME, temp_out.extent(0));
    netCDF::NcDim latDim = temperatureFile->addDim(LAT_NAME, temp_out.extent(1));
    netCDF::NcDim lonDim = temperatureFile->addDim(LON_NAME, temp_out.extent(2));

    // Define the coordinate variables (but, for some reason, not for the levels).
    netCDF::NcVar latVar = temperatureFile->addVar(LAT_NAME, netCDF::ncFloat, latDim);
    netCDF::NcVar lonVar = temperatureFile->addVar(LON_NAME, netCDF::ncFloat, lonDim);

    // Define units attributes for coordinate vars. This attaches a
    // text attribute to each of the coordinate variables, containing the units.
    latVar.putAtt(UNITS_NAME, DEGREES_NORTH);
    lonVar.putAtt(UNITS_NAME, DEGREES_EAST);

    // Define the netCDF variables for the temperature data.
    netCDF::NcVar tempVar  = temperatureFile->addVar(TEMP_NAME, netCDF::ncFloat,
                                                     {recDim, lvlDim, latDim, lonDim});

    // Define units attributes for coordinate vars. This attaches a
    // text attribute to each of the coordinate variables, containing
    // the units.
    tempVar.putAtt(UNITS_NAME, TEMP_UNITS);

    // Write the coordinate variable data to the file.
    latVar.putVar(lats.data());
    lonVar.putVar(lons.data());

    // Write the pretend data.  The arrays only hold one timestep worth
    // of data. We will just rewrite the same data for each timestep.
    for (int rec: xrange(NREC)) 
        tempVar.putVar({rec,0,0,0},
                       {1,temp_out.extent(0),temp_out.extent(1),temp_out.extent(2)},
                       temp_out.data());

    // Note: When the unique pointer temperatureFile does out of scope
    // or is reset to a nullptr, the file is automatically closed.
    temperatureFile.reset(nullptr);
    
    std::cout << "*** SUCCESS writing example file " << FILE_NAME << "!\n";
    return 0;
}
