//
// mpisin.cpp
//
// Writes a sine function on the interval [0,2*pi] to a file in parallel
//
// Illustrates domain decomposition and distributed memory
//
// Compile with
//
//  mpicxx mpisin.cpp -o mpisin
//
// Run with
//
//  mpirun -n S ./mpisin N F
//
// where S is the number of processes to use,
//       N is the number of point to produce,
//       F is the name of the file to write the sine function into.
//
// Ramses van Zon, University of Toronto, 2022

#include <iostream>
#include <mpi.h>
#include <rarray>

int main(int argc, char* argv[])
{
    // initialize mpi
    int rank, size;
    MPI_Init(&argc, &argv);
    MPI_Comm_size(MPI_COMM_WORLD, &size);
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);

    // check and interpret command line arguments
    if (argc!=3) MPI_Abort(MPI_COMM_WORLD,1);
    long N = atol(argv[1]);
    char* filename = argv[2];

    // split up the domain of N points, such that each process holds Nlocal
    int Nlocal = N / size;         // this may round down!
    int ibegin = rank * Nlocal;    // first index for this rank
    int iend = (rank+1) * Nlocal;  // end index for this rank, i.e., last index + 1
    // except if N is not divisable by size, adjust size of the last rank
    if (rank == size-1) {
        iend = N;
        Nlocal = iend - ibegin;
    }

    // allocate and generate the array
    rvector<double> sines(Nlocal); // there's no global array of size N !
    double x1 = 0.0;
    double x2 = 2*3.14159;
    for (int i = 0; i < Nlocal; i++) {
        double x = x1 + (x2-x1) * (i+ibegin)/N;
        sines[i] = sin(x);
    }

    // write out with MPI-IO (just as an example)
    MPI_File fh;
    MPI_File_open(MPI_COMM_WORLD, filename,
                  MPI_MODE_WRONLY|MPI_MODE_CREATE,
                  MPI_INFO_NULL, &fh);
    long offset = sizeof(double)*ibegin;
    MPI_File_write_at_all(fh, offset, sines, Nlocal, MPI_DOUBLE, MPI_STATUS_IGNORE);
    MPI_File_close(&fh);

    // all done
    MPI_Finalize();
}        
