/* 
   Temp4dWr.cpp - example of writing to netcdf using the modern c++ interface.
   Based on pres_temp_4D_wr.cpp from the unidata.ucar netCDF package. 
   Shortened, writing only temperature and not pressure, and made more c++-like.

   Original copyright notice:

   This is part of the netCDF package.
   Copyright 2006 University Corporation for Atmospheric Research/Unidata.
   See COPYRIGHT file for conditions of use.

   This is an example program which writes some 4D pressure and
   temperatures. This example demonstrates the netCDF C++ API. 

   This is part of the netCDF tutorial:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-tutorial

   Full documentation of the netCDF C++ API can be found at:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-cxx

   $Id: pres_temp_4D_wr.cpp,v 1.6 2010/02/11 22:36:42 russ Exp $
*/

#include <netcdf>
#include <iostream>
#include <rarray>
#include "Temp4dparameters.h"

using namespace std;
using namespace netCDF;

int main()
{
   // We will write latitude and longitude fields. 
   rarray<float,1> lats(NLAT);
   rarray<float,1> lons(NLON);

   // Program variables to hold the data we will write out. We will
   // only need enough space to hold one timestep of data; one record.
   rarray<float,3> temp_out(NLVL,NLAT,NLON);

   int i=0;  //used in the data generation loop
  
   // create some pretend data. If this wasn't an example program, we
   // would have some real data to write for example, model output.
   for (int lat = 0; lat < NLAT; lat++) {
      lats[lat] = START_LAT + 5. * lat;
   }
   for (int lon = 0; lon < NLON; lon++) {
      lons[lon] = START_LON + 5. * lon;
   }

   for (int lvl = 0; lvl < NLVL; lvl++) {
      for (int lat = 0; lat < NLAT; lat++) {
         for (int lon = 0; lon < NLON; lon++) {
            temp_out[lvl][lat][lon] = SAMPLE_TEMP + i++;
         }
      }
   }
   
   // Create the file.
   NcFile test(FILE_NAME, NcFile::replace);

   // Define the dimensions. NetCDF will hand back an ncDim object for
   // each.
   NcDim recDim = test.addDim(REC_NAME);  //adds an unlimited dimension
   NcDim lvlDim = test.addDim(LVL_NAME, NLVL);
   NcDim latDim = test.addDim(LAT_NAME, NLAT);
   NcDim lonDim = test.addDim(LON_NAME, NLON);

   // Define the coordinate variables.
   NcVar latVar = test.addVar(LAT_NAME, ncFloat, latDim);
   NcVar lonVar = test.addVar(LON_NAME, ncFloat, lonDim);

   // Define units attributes for coordinate vars. This attaches a
   // text attribute to each of the coordinate variables, containing
   // the units.
   latVar.putAtt(UNITS, DEGREES_NORTH);
   lonVar.putAtt(UNITS, DEGREES_EAST);

   // Define the netCDF variables for the temperature data.
   vector<NcDim> dimVector{recDim, lvlDim, latDim, lonDim};
   NcVar tempVar  = test.addVar(TEMP_NAME, ncFloat, dimVector);

   // Define units attributes for coordinate vars. This attaches a
   // text attribute to each of the coordinate variables, containing
   // the units.
   tempVar.putAtt(UNITS, TEMP_UNITS);

   // Write the coordinate variable data to the file.
   latVar.putVar(lats.data());
   lonVar.putVar(lons.data());

   // Write the pretend data. This will write our surface temperature
   // data. The arrays only hold one timestep worth of data. We will
   // just rewrite the same data for each timestep. In a real
   // application, the data would change between timesteps.
   for (size_t rec = 0; rec < NREC; rec++) {
      tempVar.putVar({rec,0,0,0}, {1,NLVL,NLAT,NLON}, temp_out.data());
   }

   // The file is automatically closed by the destructor. This frees
   // up any internal netCDF resources associated with the file, and
   // flushes any buffers.

   cout << "*** SUCCESS writing example file " << FILE_NAME << "!" << endl;

   return 0;
}
