/* 
   Temp4dWr.cpp - example of reading from a netcdf file using the modern c++ interface.
   Based on pres_temp_4D_rd.cpp from the unidata.ucar netCDF package. 
   Shortened, reading only temperature and not pressure, and made more c++-like.

   Original copyright notice:

   This is part of the netCDF package.
   Copyright 2006 University Corporation for Atmospheric Research/Unidata.
   See COPYRIGHT file for conditions of use.

   This is an example which reads some 4D pressure and temperature
   values. The data file read by this program is produced by the
   companion program pres_temp_4D_wr.cpp. It is intended to illustrate
   the use of the netCDF C++ API.

   This program is part of the netCDF tutorial:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-tutorial

   Full documentation of the netCDF C++ API can be found at:
   http://www.unidata.ucar.edu/software/netcdf/docs/netcdf-cxx

   $Id: pres_temp_4D_rd.cpp,v 1.5 2010/02/11 22:36:42 russ Exp $
*/

#include <iostream>
#include <netcdf>
#include <rarray>
#include "Temp4dparameters.h"

using namespace std;
using namespace netCDF;

// Return this code to the OS in case of failure.
static const int NC_ERR = 2;

int main()
{
   // These arrays will store the latitude and longitude values.
   rarray<float,1> lats(NLAT);
   rarray<float,1> lons(NLON);
   
   // These arrays will hold the data we will read in. We will only
   // need enough space to hold one timestep of data; one record.
   rarray<float,3> temp_in(NLVL,NLAT,NLON);

   NcFile dataFile(FILE_NAME, NcFile::read);

   // Get the latitude and longitude variables and read data.
   NcVar latVar = dataFile.getVar(LAT_NAME);
   if (latVar.isNull()) return NC_ERR;

   NcVar lonVar = dataFile.getVar(LON_NAME);
   if (lonVar.isNull()) return NC_ERR;

   lonVar.getVar(lons.data());
   latVar.getVar(lats.data());

   // Check the coordinate variable data. 
   for (int lat = 0; lat < NLAT; lat++) {
      if (lats[lat] != START_LAT + 5. * lat) {
         return NC_ERR;
      }
   }

   for (int lon = 0; lon < NLON; lon++) {
      if (lons[lon] != START_LON + 5. * lon) {
         return NC_ERR;
      }
   }
   
   // Get the temperature variables and read data one time step at a time
   NcVar tempVar = dataFile.getVar(TEMP_NAME);
   if (tempVar.isNull()) {
      return NC_ERR; 
   }

   for (size_t rec = 0; rec < NREC; rec++) {

      // Read the data one record at a time.
      tempVar.getVar({rec,0,0,0}, {1,NLVL,NLAT,NLON}, temp_in.data());
     
      int i=0;  //used in the data generation loop
      for (int lvl = 0; lvl < NLVL; lvl++) {
         for (int lat = 0; lat < NLAT; lat++) {
            for (int lon = 0; lon < NLON; lon++) {
               if (temp_in[lvl][lat][lon] != (float)(SAMPLE_TEMP + i++)) {
                  return NC_ERR;
               }
            }
         }
      }
      
   } // next record 
       
   // The file is automatically closed by the destructor. This frees
   // up any internal netCDF resources associated with the file, and
   // flushes any buffers.

   cout << "*** SUCCESS reading example file pres_temp_4D.nc!" << endl;

   return 0;

}
